/* Copyright (C) 2016-2018 RealVNC Ltd.  All Rights Reserved.
 */

/* This is sample code intended to demonstrate part of the
 * VNC Mobile Solution SDK. It is not intended as a production-ready
 * component.
 */

#include "BaseDecoder.h"
#include "BaseDecoderLogHandler.h"
#include "BaseDecoderUtils.h"
#include "VNCCommonDecoderPluginContextImpl.h"

#include <vnccommon/OptionalAutoPtr.h>

#include <cstring>

namespace
{
    vnc_bool_t convertBoolToVncBool(const bool value)
    {
        if(value)
        {
            return vnc_true;
        }
        else
        {
            return vnc_false;
        }
    }

    extern "C"
    {
        void VNCCALL
        vncCommonDecoderCallbackDestroyedImpl(
                VNCCommonDecoderPluginContext pluginContext)
        {
            pluginContext->callbackDestroy();
            delete pluginContext;
        }

        void VNCCALL
        vncCommonDecoderCallbackStreamCreatedImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const vnc_uint64_t streamId,
                const VNCCommonDecoderMediaType mediaType,
                const VNCCommonDecoderStreamSpecificType specificType)
        {
            pluginContext->callbackStreamCreatedImpl(
                    streamId,
                    mediaType,
                    specificType);
        }

        VNCCommonDecoderError VNCCALL
        vncCommonDecoderCallbackQueryEncodingSupportAudioLPCMImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const VNCCommonDecoderAudioFormatDetailsLPCM* formatDetails,
                vnc_bool_t *const out_isEncodingSupported,
                VNCCommonDecoderAudioFormatDetailsLPCM* *const out_suggestedFormatDetailsList,
                vnc_size_t *const out_suggestedFormatDetailsCount)
        {
            return pluginContext->callbackQueryEncodingSupportAudioLPCMImpl(
                    formatDetails,
                    out_isEncodingSupported,
                    out_suggestedFormatDetailsList,
                    out_suggestedFormatDetailsCount);
        }

        void VNCCALL
        vncCommonDecoderCallbackStreamAudioContentHintImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const vnc_uint64_t streamId,
                const VNCCommonDecoderAudioStreamContentType *const audioStreamContentTypeList,
                const vnc_size_t audioStreamContentTypeCount)
        {
            pluginContext->callbackStreamAudioContentHintImpl(
                    streamId,
                    audioStreamContentTypeList,
                    audioStreamContentTypeCount);
        }

        void VNCCALL
        vncCommonDecoderCallbackStreamAudioEncodingHintLPCMImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const vnc_uint64_t streamId,
                const VNCCommonDecoderAudioFormatDetailsLPCM *const formatDetails)
        {
            pluginContext->callbackStreamAudioEncodingHintLPCMImpl(
                    streamId,
                    formatDetails);
        }

        VNCCommonDecoderError VNCCALL
        vncCommonDecoderCallbackQueryEncodingSupportVideoH264Impl(
                VNCCommonDecoderPluginContext pluginContext,
                const vnc_size_t h264AttributesSize,
                const VNCCommonDecoderH264Attributes *const h264Attributes,
                const VNCCommonDecoderVideoMode *const videoModeList,
                const vnc_size_t videoModeCount,
                vnc_bool_t *const out_isEncodingSupportedList,
                VNCCommonDecoderVideoMode* *const out_suggestedVideoModeList,
                vnc_size_t *const out_suggestedVideoModeCount)
        {
            return pluginContext->callbackQueryEncodingSupportVideoH264Impl(
                    h264AttributesSize,
                    h264Attributes,
                    videoModeList,
                    videoModeCount,
                    out_isEncodingSupportedList,
                    out_suggestedVideoModeList,
                    out_suggestedVideoModeCount);
        }

        void VNCCALL
        vncCommonDecoderCallbackStreamVideoEncodingHintH264Impl(
                VNCCommonDecoderPluginContext pluginContext,
                const vnc_uint64_t streamId,
                const vnc_size_t h264AttributesSize,
                const VNCCommonDecoderH264Attributes *const h264Attributes,
                const VNCCommonDecoderVideoMode *const videoMode)
        {
            pluginContext->callbackStreamVideoEncodingHintH264Impl(
                    streamId,
                    h264AttributesSize,
                    h264Attributes,
                    videoMode);
        }

        void VNCCALL
        vncCommonDecoderCallbackStreamStartedImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const vnc_uint64_t streamId)
        {
            pluginContext->callbackStreamStartedImpl(streamId);
        }

        void VNCCALL
        vncCommonDecoderCallbackStreamStoppedImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const vnc_uint64_t streamId,
                const vnc_bool_t stopImmediately)
        {
            pluginContext->callbackStreamStoppedImpl(streamId, stopImmediately);
        }

        void VNCCALL
        vncCommonDecoderCallbackStreamDestroyedImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const vnc_uint64_t streamId)
        {
            pluginContext->callbackStreamDestroyedImpl(streamId);
        }

        void VNCCALL
        vncCommonDecoderCallbackStreamPayloadImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const vnc_uint64_t streamId,
                const vnc_size_t payloadStructSize,
                VNCCommonDecoderStreamPayload *const payload)
        {
            pluginContext->callbackStreamPayloadImpl(
                    streamId,
                    payloadStructSize,
                    payload);
        }

        void VNCCALL
        vncCommonDecoderCallbackStreamMuteEnableImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const vnc_uint64_t streamId)
        {
            pluginContext->callbackStreamMuteEnableImpl(streamId);
        }

        void VNCCALL
        vncCommonDecoderCallbackStreamMuteDisableImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const vnc_uint64_t streamId)
        {
            pluginContext->callbackStreamMuteDisableImpl(streamId);
        }

        void VNCCALL
        vncCommonDecoderCallbackStreamDuckEnableImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const vnc_uint64_t streamId,
                const vnc_uint64_t suggestedRampMs,
                const vnc_uint32_t suggestedAttenuationDb)
        {
            pluginContext->callbackStreamDuckEnableImpl(
                    streamId,
                    suggestedRampMs,
                    suggestedAttenuationDb);
        }

        void VNCCALL
        vncCommonDecoderCallbackStreamDuckDisableImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const vnc_uint64_t streamId)
        {
            pluginContext->callbackStreamDuckDisableImpl(streamId);
        }

        void VNCCALL
        vncCommonDecoderCallbackStreamFrameRateLimitRequestSuccessImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const vnc_uint64_t streamId,
                const vnc_uint32_t newFrameRate)
        {
            pluginContext->callbackStreamFrameRateLimitRequestSuccessImpl(
                    streamId,
                    newFrameRate);
        }

        VNCCommonDecoderError VNCCALL
        vncCommonDecoderCallbackPropertySetImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const VNCCommonDecoderPropertyKey propertyKey,
                const VNCCommonDecoderDataType propertyType,
                void *const propertyValue)
        {
            return pluginContext->callbackPropertySetImpl(
                    propertyKey,
                    propertyType,
                    propertyValue);
        }

        void VNCCALL
        vncCommonDecoderCallbackGetEventHandlesImpl(
                VNCCommonDecoderPluginContext pluginContext,
                VNCCommonDecoderEventHandleParams* *const out_eventHandleList,
                vnc_size_t *const out_eventHandleCount)
        {
            pluginContext->callbackGetEventHandlesImpl(
                    out_eventHandleList,
                    out_eventHandleCount);
        }

        void VNCCALL
        vncCommonDecoderCallbackEventHandleActivityImpl(
                VNCCommonDecoderPluginContext pluginContext,
                const VNCCommonDecoderEventHandle *const activeEventHandleList,
                const size_t activeEventHandleCount)
        {
            pluginContext->callbackEventHandleActivityImpl(
                    activeEventHandleList,
                    activeEventHandleCount);
        }

        const VNCCommonDecoderCallbacks staticCallbacks =
        {
            vncCommonDecoderCallbackDestroyedImpl,
            vncCommonDecoderCallbackStreamCreatedImpl,
            vncCommonDecoderCallbackQueryEncodingSupportAudioLPCMImpl,
            vncCommonDecoderCallbackStreamAudioContentHintImpl,
            vncCommonDecoderCallbackStreamAudioEncodingHintLPCMImpl,
            vncCommonDecoderCallbackQueryEncodingSupportVideoH264Impl,
            vncCommonDecoderCallbackStreamVideoEncodingHintH264Impl,
            vncCommonDecoderCallbackStreamStartedImpl,
            vncCommonDecoderCallbackStreamStoppedImpl,
            vncCommonDecoderCallbackStreamDestroyedImpl,
            vncCommonDecoderCallbackStreamPayloadImpl,
            vncCommonDecoderCallbackStreamMuteEnableImpl,
            vncCommonDecoderCallbackStreamMuteDisableImpl,
            vncCommonDecoderCallbackStreamDuckEnableImpl,
            vncCommonDecoderCallbackStreamDuckDisableImpl,
            vncCommonDecoderCallbackStreamFrameRateLimitRequestSuccessImpl,
            vncCommonDecoderCallbackPropertySetImpl,
            vncCommonDecoderCallbackGetEventHandlesImpl,
            vncCommonDecoderCallbackEventHandleActivityImpl
        };
    }
}

BaseDecoderImpl::BaseDecoderImpl()
{
}

BaseDecoderImpl::~BaseDecoderImpl()
{
}

extern "C" VNCCommonDecoderPluginContextImpl* VNCCALL
VNCCommonDecoderInitialize(
        VNCCommonDecoderFrameworkContext decoderFrameworkContext,
        VNCCommonDecoderApplicationContext applicationContext,
        VNCCommonDecoderCallbacks *const out_decoderCallbacks,
        const size_t decoderCallbacksSize,
        const VNCCommonDecoderSupportingAPI *const decoderSupportingAPI,
        const size_t decoderSupportingAPISize,
        const VNCCommonDecoderSDKAttributes* sdkAttributes,
        size_t sdkAttributesSize,
        VNCCommonDecoderSupportInfo *const out_supportInfo,
        const size_t supportInfoSize,
        VNCCommonDecoderCreationParams *const out_creationParameters,
        const size_t creationParametersSize)
{
    try
    {
        if(out_decoderCallbacks == NULL
                || decoderSupportingAPI == NULL
                || out_supportInfo == NULL
                || out_creationParameters == NULL)
        {
            return NULL;
        }

        VNCCommonDecoderSupportingAPI supportingApi;
        ::memset(&supportingApi, 0, sizeof(VNCCommonDecoderSupportingAPI));
        VNCCOMMON_SAFE_MEMCPY_STRUCT(
                VNCCommonDecoderSupportingAPI,
                decoderSupportingAPISize,
                VNCCommonDecoderSupportingAPIStreamDestroyCompleted*,
                vncCommonDecoderSupportingAPIStreamDestroyCompleted,
                &supportingApi,
                decoderSupportingAPI);

        ::memset(out_decoderCallbacks, 0, decoderCallbacksSize);
        VNCCOMMON_SAFE_MEMCPY_STRUCT(
                VNCCommonDecoderCallbacks,
                decoderCallbacksSize,
                VNCCommonDecoderCallbackEventHandleActivity*,
                vncCommonDecoderCallbackEventHandleActivity,
                out_decoderCallbacks,
                &staticCallbacks);

        VNCCommonDecoderSDKAttributes sdkAttributesSafe;
        ::memset(&sdkAttributesSafe, 0, sizeof(sdkAttributesSafe));
        ::memcpy(&sdkAttributesSafe, sdkAttributes, sdkAttributesSize);

        std::auto_ptr<VNCCommonDecoderPluginContextImpl> decoderContext(
                new VNCCommonDecoderPluginContextImpl(
                        supportingApi,
                        decoderFrameworkContext));

        vnccommon::LogHandler::Tagged log
                = decoderContext->support().log().tag("BaseDecoder::VNCCommonDecoderInitialize");

        {
            std::auto_ptr<BaseDecoderImpl> decoderImpl = instantiateDecoder(
                    applicationContext,
                    sdkAttributesSafe,
                    decoderContext->support());

            decoderContext->setImpl(decoderImpl);
        }

        ::memset(out_supportInfo, 0, supportInfoSize);
        DecoderSupportInfoDeallocator supportInfoDeallocator(
                decoderContext->support(),
                *out_supportInfo,
                supportInfoSize);

        if(VNCCOMMON_DOES_STRUCT_CONTAIN_MEMBER(
                VNCCommonDecoderSupportInfo,
                supportInfoSize,
                char*,
                decoderName))
        {
            const std::string decoderName
                    = decoderContext->getSupportInfoDecoderName();

            out_supportInfo->decoderName
                    = decoderContext->support()
                            .supportingApiAllocString(decoderName.c_str());

            if(out_supportInfo->decoderName == NULL)
            {
                log.error("Failed to allocate decoder name");

                // Allocation failed! RAII will deal with the cleanup.
                return NULL;
            }
        }

        if(VNCCOMMON_DOES_STRUCT_CONTAIN_MEMBER(
                VNCCommonDecoderSupportInfo,
                supportInfoSize,
                char*,
                decoderVersion))
        {
            const std::string decoderVersion
                    = decoderContext->getSupportInfoDecoderVersion();

            out_supportInfo->decoderVersion
                    = decoderContext->support()
                            .supportingApiAllocString(decoderVersion.c_str());

            if(out_supportInfo->decoderVersion == NULL)
            {
                log.error("Failed to allocate decoder version");

                // Allocation failed! RAII will deal with the cleanup.
                return NULL;
            }
        }

        ::memset(out_creationParameters, 0, creationParametersSize);

        if(VNCCOMMON_DOES_STRUCT_CONTAIN_MEMBER(
                VNCCommonDecoderCreationParams,
                creationParametersSize,
                vnc_bool_t,
                disableBuffering))
        {
            out_creationParameters->disableBuffering
                    = convertBoolToVncBool(
                            decoderContext->getCreationParamDisableBuffering());
        }

        // No more errors
        supportInfoDeallocator.release();

        log.info("Successfully created decoder.");

        return decoderContext.release();
    }
    catch(const std::exception& e)
    {
        if(decoderSupportingAPI
                && decoderSupportingAPI->vncCommonDecoderSupportingAPILog)
        {
            decoderSupportingAPI->vncCommonDecoderSupportingAPILog(
                    decoderFrameworkContext,
                    0, // Severity
                    "BaseDecoder Unhandled Exception",
                    e.what());
        }
    }
    catch(...)
    {
        if(decoderSupportingAPI
                && decoderSupportingAPI->vncCommonDecoderSupportingAPILog)
        {
            decoderSupportingAPI->vncCommonDecoderSupportingAPILog(
                    decoderFrameworkContext,
                    0, // Severity
                    "BaseDecoder Unhandled Exception",
                    "Unknown exception in BaseDecoder init");
        }
    }

    return NULL;
}

